import cv2
import numpy as np
import matplotlib.pyplot as plt
from scipy.spatial import ConvexHull
import tkinter as tk
from tkinter import filedialog
from tkinter import simpledialog
from tkinter import messagebox
from PIL import Image, ImageTk # install as pillow

# Global variables for storing the coordinates of the polygon
polygon_points = []
cell_analysis_results = []

def select_roi(event, x, y, flags, param):
    global polygon_points, image, clone
    if event == cv2.EVENT_LBUTTONDOWN:
        polygon_points.append((x, y))
    elif event == cv2.EVENT_LBUTTONUP:
        # Draw polygon segments with higher visibility
        if len(polygon_points) > 1:
            overlay = clone.copy()
            cv2.polylines(overlay, np.array([polygon_points]), False, (0, 255, 0), 2)  # Green outline
            cv2.addWeighted(overlay, 0.7, clone, 0.3, 0, clone)  # Blend overlay with clone image
        cv2.imshow("image", clone)

def display_image(image, title):
    height, width = image.shape[:2]
    aspect_ratio = width / height
    plt.figure(figsize=(10 * aspect_ratio, 10))
    plt.title(title)
    plt.imshow(cv2.cvtColor(image, cv2.COLOR_BGR2RGB))
    plt.axis('off')
    plt.show()
    print(f"{title} size: {width} x {height}")

def preprocess_image(image):
    # Convert to grayscale
    gray = cv2.cvtColor(image, cv2.COLOR_BGR2GRAY)

    # Apply Gaussian blur
    blurred = cv2.GaussianBlur(gray, (5, 5), 0)

    # Apply thresholding
    _, binary = cv2.threshold(blurred, 0, 255, cv2.THRESH_BINARY_INV + cv2.THRESH_OTSU)

    return binary

def analyze_cell(binary_image, original_image, pixel_scale):
    global cell_analysis_results
    # Find contours
    contours, _ = cv2.findContours(binary_image, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
    
    if contours:
        for contour in contours:
            # Ensure contour has enough points for convex hull
            if len(contour) < 3:
                print(f"Skipping contour with insufficient points: {len(contour)}")
                continue

            # Create a mask for the selected cell
            mask = np.zeros_like(original_image[:, :, 0], dtype=np.uint8)  # Create mask from single channel
            cv2.drawContours(mask, [contour], -1, (255), thickness=cv2.FILLED)

            # Apply the mask to the original image to isolate the cell
            cell_image = cv2.bitwise_and(original_image, original_image, mask=mask)

            # Analyze the selected cell
            area = cv2.contourArea(contour)
            perimeter_length = cv2.arcLength(contour, True)

            # Calculate convex hull if possible
            try:
                hull = ConvexHull(contour.reshape(-1, 2))
                convex_area = hull.volume
                solidity = area / convex_area
            except Exception as e:
                print(f"Convex hull calculation failed. Skipping. Error: {e}")
                continue

            # Store cell metrics with physical units
            cell_analysis_results.append({
                'Area': area / (pixel_scale ** 2),  # Convert to square units (e.g., sq. inches or sq. millimeters)
                'Perimeter': perimeter_length / pixel_scale,  # Convert to units (e.g., inches or millimeters)
                'Convex Area': convex_area / (pixel_scale ** 2),  # Convert to square units
                'Solidity': solidity  # Dimensionless
            })

            # Draw contour on a separate image (clone) to overlay on the original image
            cv2.drawContours(clone, [contour], -1, (0, 0, 255), 2)  # Red outline
            cv2.drawContours(clone, [contour], -1, (255, 0, 0), thickness=cv2.FILLED)  # Blue filled area

    else:
        print('No contours found in the selected region.')

def calculate_average_metrics(cell_metrics):
    areas = [cell['Area'] for cell in cell_metrics]
    perimeters = [cell['Perimeter'] for cell in cell_metrics]
    convex_areas = [cell['Convex Area'] for cell in cell_metrics]
    solidities = [cell['Solidity'] for cell in cell_metrics]
    
    avg_area = np.mean(areas)
    avg_perimeter = np.mean(perimeters)
    avg_convex_area = np.mean(convex_areas)
    avg_solidity = np.mean(solidities)
    
    return avg_area, avg_perimeter, avg_convex_area, avg_solidity

def main(image_path, pixel_scale):
    global image, clone, polygon_points, cell_analysis_results
    # Read the image
    image = cv2.imread(image_path)
    clone = image.copy()
    polygon_points = []  # Initialize polygon points
    cell_analysis_results = []  # Initialize analysis results

    cv2.namedWindow("image")
    cv2.setMouseCallback("image", select_roi)

    # Display the original image
    display_image(image, 'Original Image (Close to proceed)')
    print("Instructions: Click to select points. Press 'c' to confirm selection.")
    
    # Display the image and wait for the user to select a region
    while True:
        cv2.imshow("image", image)
        key = cv2.waitKey(1) & 0xFF
        if key == ord("r"):
            image = clone.copy()
            polygon_points = []  # Reset polygon points
        elif key == ord("c"):
            break

    if len(polygon_points) >= 3:
        # Create a mask for the selected polygon
        mask = np.zeros_like(clone[:, :, 0], dtype=np.uint8)
        cv2.fillPoly(mask, [np.array(polygon_points)], (255, 255, 255))

        # Apply the mask to get the ROI
        roi = cv2.bitwise_and(clone, clone, mask=mask)

        # Display the original image with selected ROI contours
        display_image(clone, 'Original Image with Selected ROI')

        # Analyze the selected ROI with contours and metrics
        analyze_cell(mask, clone, pixel_scale)

        # Calculate average metrics
        avg_area, avg_perimeter, avg_convex_area, avg_solidity = calculate_average_metrics(cell_analysis_results)

        # Print results with physical units
        print(f'Average Metrics for Cells:')
        print(f'  Average Area: {avg_area} square units')
        print(f'  Average Perimeter: {avg_perimeter} units')
        print(f'  Average Convex Area: {avg_convex_area} square units')
        print(f'  Average Solidity: {avg_solidity}')

        # Display image with contours
        display_image(clone, 'Image with Contours')

    cv2.destroyAllWindows()

class ImageAnalyzerApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Cell Image Analyzer")

        self.label = tk.Label(root, text="Cell Image Analyzer", font=("Helvetica", 16))
        self.label.pack(pady=10)

        self.upload_button = tk.Button(root, text="Upload Image", command=self.upload_image)
        self.upload_button.pack(pady=10)

        self.pixel_scale_label = tk.Label(root, text="Pixel Scale:")
        self.pixel_scale_label.pack(pady=5)

        self.pixel_scale_entry = tk.Entry(root)
        self.pixel_scale_entry.pack(pady=5)
        self.pixel_scale_entry.insert(0, "300")

        self.analyze_button = tk.Button(root, text="Analyze", command=self.analyze_image)
        self.analyze_button.pack(pady=10)

        self.image_label = tk.Label(root)
        self.image_label.pack(pady=10)

        self.image_path = None

    def upload_image(self):
        self.image_path = filedialog.askopenfilename()
        if self.image_path:
            image = Image.open(self.image_path)
            image.thumbnail((400, 400))
            photo = ImageTk.PhotoImage(image)
            self.image_label.config(image=photo)
            self.image_label.image = photo

    def analyze_image(self):
        if not self.image_path:
            messagebox.showerror("Error", "Please upload an image first.")
            return

        pixel_scale = self.pixel_scale_entry.get()
        if not pixel_scale.isdigit():
            messagebox.showerror("Error", "Please enter a valid pixel scale.")
            return

        pixel_scale = int(pixel_scale)
        main(self.image_path, pixel_scale)

if __name__ == "__main__":
    root = tk.Tk()
    app = ImageAnalyzerApp(root)
    root.mainloop()


# In[ ]:




